function [exitFlag exitMsg netStruct] = ba_networkUtilityMaximization(netStruct, algorithmParameters, net2planParameters)

    % ba_networkUtilityMaximization
    %
	% Algorithm for bandwidth assignment problems which solves the
    % Network Utility Maximization (NUM) problem giving an alpha-fair
    % bandwidth for traffic demands.
    %
    % It requires CVX solver installed and running.
    %
    % PARAM: alpha | 1 | Parameter to define utility function (alpha>=0)

    try
        %% Sanity checks
        alpha = checkAlgorithmParameters(algorithmParameters);
        assert(isfield(netStruct, 'linkTable') && ~isempty(netStruct.linkTable), 'A link table is required');
        assert(isfield(netStruct, 'demandTable') && ~isempty(netStruct.demandTable), 'A demand table is required');
        assert(isCvxInstalledAndRunning(), 'CVX solver must be installed and running');
        
        E = size(netStruct.linkTable, 1);
        D = size(netStruct.demandTable, 1);
        
        x_dp = eye(D,D);
        pathList = cell(1, D);
        for demandId = 1:D
            sequenceOfLinks = shortestPath(netStruct.linkTable, ones(1, E), ...
                netStruct.demandTable(demandId, 1), ...
                netStruct.demandTable(demandId, 2));
            
            assert(~isempty(sequenceOfLinks), sprintf('No feasible path for demand %d', demandId));
            pathList{demandId} = sequenceOfLinks;
        end
        
        u_e = netStruct.linkCapacityInErlangs;
        h_d = netStruct.offeredTrafficInErlangs;
        delta_pe = seqLinksPerPath2linkOccupancyPerPath(pathList, E);

        cvx_begin

            cvx_quiet(true);

            % Variables
            variables r_d(1,D);
            dual variables v_d w_d pi_e;

            % Objective function
            if alpha == 0
                maximize(sum(r_d))
            elseif alpha == 1
                maximize(sum(log(r_d)))
            else
                maximize(((1-alpha)^-1) * sum(pow_p(r_d , 1-alpha)))
            end
        
            % Constraints
            subject to
                v_d:                    r_d >= 0;   % Non-negativity constraints (D)
                pi_e: r_d * x_dp * delta_pe <= u_e; % Capacity constraints (E)

        cvx_end
        
        %% Check results
        PRECISIONFACTOR = 1e-3;
        
        if strcmp(cvx_status, 'Solved')
        
            r_d = min(h_d, r_d);

            % Complementary slackness
            assert( all( abs(v_d .* r_d) < PRECISIONFACTOR ), 'Complementary slackness: Non-negativity constraints violated');
            assert( all( abs(pi_e .* (u_e - r_d*x_dp*delta_pe)) < PRECISIONFACTOR ), 'Complementary slackness: Capacity constraints violated');

            netStruct.offeredTrafficInErlangs = r_d;
            netStruct.pathList = pathList;
            netStruct.routingMatrix = x_dp;

            exitFlag = 0;
            exitMsg = 'OK';
            
        else
            
            exitFlag = -1;
            exitMsg = cvx_status;
            netStruct = [];
            
        end
        
    catch e
        
        exitFlag = -1;
        exitMsg = e.message;
        netStruct = [];
        
    end

 end

 function alpha = checkAlgorithmParameters(algorithmParameters)
 
    try
        assert(isfield(algorithmParameters, 'alpha') && ~isempty(algorithmParameters.alpha), '"alpha" parameter is required');
        alpha = str2double(algorithmParameters.alpha);
        assert(isfinite(alpha) && alpha >= 0, '"alpha" must be greater or equal than zero');
    catch e
        error('%s: %s', 'checkAlgorithmParameters', e.message);
    end
 
 end