function [exitFlag exitMsg netStruct] = ca_netMinCongestion(netStruct, algorithmParameters, net2planParameters)

    % ca_netMinCongestion
    %
    % Algorithm for capacity assignment problems which tries to minimize
    % network congestion.
    %
    % It requires CVX solver installed and running.
    %
    % PARAM: alpha | 0 | Linear cost coefficient with capacity
	% PARAM: beta | 1 | Linear cost coefficient with capacity-distance power product
	% PARAM: gamma | 0.5 | Distance power
	% PARAM: C_max | 10000 | Maximum network cost ( C = sum_{e in E} (u_e * (alpha + beta * l(e)^gamma) )
    
    try
		%% Sanity checks
        [alpha, beta, gamma, C_max] = checkAlgorithmParameters(algorithmParameters);
		assert(isfield(netStruct, 'nodeXYPositionTable') && isfield(netStruct, 'linkTable') && ...
			isfield(netStruct, 'demandTable') && isfield(netStruct, 'pathList') && ...
            isfield(netStruct, 'routingMatrix'), ...
			'A physical topology (nodes + links), a demand set and a routing are required');
        assert(isCvxInstalledAndRunning(), 'CVX solver must be installed and running');
			
        E = size(netStruct.linkTable, 1);
        
		h_d = netStruct.offeredTrafficInErlangs;
        x_dp = netStruct.routingMatrix;
        delta_pe = seqLinksPerPath2linkOccupancyPerPath(netStruct.pathList, E);
        
        y_e = trafficPerLink(h_d, x_dp, delta_pe);
        distance_e = netStruct.linkLengthInKm;
        
        cvx_begin
        
            cvx_quiet(true);
            variables rho rho_e(1, E);
            dual variables mult_nonNeg mult_lt1 mult_cost;
            minimize( rho );
            subject to
            
                rho <= 1;
                mult_lt1: rho_e <= rho;
                mult_nonNeg: rho_e >= 0;
                mult_cost: sum(inv_pos(rho_e) .* y_e .* (alpha + beta * (distance_e .^ gamma))) <= C_max;
                
        cvx_end
        
        if strcmp(cvx_status, 'Solved') || strcmp(cvx_status, 'Inaccurate/Solved')
            exitFlag = 0;
            exitMsg = 'OK!';

            u_e = y_e ./ rho_e;
            netStruct.linkCapacityInErlangs = u_e;
        else
            exitFlag = -1;
            exitMsg = cvx_status;
        end
        
    catch e
        
        exitFlag = -1;
        exitMsg = e.message;
        netStruct = [];
        
    end

end

function [alpha, beta, gamma, C_max] = checkAlgorithmParameters(algorithmParameters)

    try
        assert(isfield(algorithmParameters, 'alpha') && ~isempty(algorithmParameters.alpha), '"alpha" is required');
        assert(isfield(algorithmParameters, 'beta') && ~isempty(algorithmParameters.beta), '"beta" is required');
        assert(isfield(algorithmParameters, 'gamma') && ~isempty(algorithmParameters.gamma), '"gamma" is required');
        assert(isfield(algorithmParameters, 'C_max') && ~isempty(algorithmParameters.C_max), '"C_max" is required');
        alpha = str2double(algorithmParameters.alpha);
        beta = str2double(algorithmParameters.beta);
        gamma = str2double(algorithmParameters.gamma);
        C_max = str2double(algorithmParameters.C_max);
        assert(isfinite(alpha), '"alpha" must be a number');
        assert(isfinite(beta), '"beta" must be a number');
        assert(isfinite(gamma), '"gamma" must be a number');
        assert(isfinite(C_max) && C_max > 0, '"C_max" must be a number greater than 0');
    catch e
        error('%s: %s', 'checkAlgorithmParameters', e.message);
    end

end
