function [exitFlag exitMsg netStruct] = fa_netMinDelayMM1(netStruct, algorithmParameters, net2planParameters)

    % fa_netMinDelayMM1
    %
    % Algorithm for flow assignment problems which tries to minimize
    % average queueing delay in the whole network.
    %
	% This algorithm has no specific parameters.
    %
    % It requires CVX solver installed and running.

    try
		%% Sanity checks
		assert(isfield(netStruct, 'nodeXYPositionTable') && isfield(netStruct, 'linkTable') && ...
			isfield(netStruct, 'demandTable'), ...
			'A physical topology (nodes + links) and a demand set are required');
			
		u_e = netStruct.linkCapacityInErlangs;
		h_d = netStruct.offeredTrafficInErlangs;
        
        assert(all(u_e > 0), 'Link capacities must be greater than 0');

        N = size(netStruct.nodeXYPositionTable, 1);
		E = numel(u_e);
		D = numel(h_d);
		A_ne = linkTable2incidenceMatrix(netStruct.linkTable, N);
		w_nd = linkTable2incidenceMatrix(netStruct.demandTable, N);
        
        cvx_begin
        
            cvx_quiet(true);
            variables x_de(D, E) rho_e(1, E);
            
            minimize( sum(quad_over_lin(rho_e, 1-rho_e, 1)) );
			
			subject to
			
                x_de <= 1;
				x_de >= 0;
				A_ne * x_de' == w_nd;
				rho_e == h_d * x_de ./ u_e;
				
        cvx_end
        
        if strcmp(cvx_status, 'Solved') || strcmp(cvx_status, 'Inaccurate/Solved')

            exitFlag = 0;
            exitMsg = 'OK!';
            
            PRECISIONFACTOR = 1e-4;
            x_de(x_de > 1-PRECISIONFACTOR) = 1;
            x_de(x_de < PRECISIONFACTOR) = 0;

            [x_dp pathList] = xde2xdp(x_de, netStruct.linkTable, ones(1, E), netStruct.demandTable);

            netStruct.routingMatrix = x_dp;
            netStruct.pathList = pathList;
            
        else
        
            exitFlag = -1;
            exitMsg = cvx_status;
        
        end
        
    catch e
        
        exitFlag = -1;
        exitMsg = e.message;
        netStruct = [];
        
    end

end