function [exitFlag exitMsg netStruct] = la_unidirectionalMinimumSpanningTree_Prim_Km(netStruct, algorithmParameters, net2planParameters)

    % la_unidirectionalMinimumSpanningTree_Prim_Km
    %
    % Algorithm for link assignment problems which generates an
    % unidirectional minimum spanning tree using Prim's algorithm and
    % link length as cost per link.
    %
    % PARAM: initialNode | 1 | Root node of the spanning tree

    try
        %% Sanity checks
        initialNode = checkAlgorithmParameters(algorithmParameters);
        assert(isfield(netStruct, 'nodeXYPositionTable'), 'A set of nodes is required');
        assert(isfield(algorithmParameters, 'initialNode'), 'An initial node is required');

        N = size(netStruct.nodeXYPositionTable, 1);
        assert(initialNode <= N, sprintf('initialNode must be between 1 and %d', N));
        
        %% Compute distance matrix
        distanceMatrix = nodeXYPositionTable2distanceMatrix(netStruct.nodeXYPositionTable);
		
		%% Perform Prim Algorithm
		[linkTable, linkCostVector] = primAlgorithm(distanceMatrix, initialNode);
		
        %% Generate netStruct
        netStruct.linkTable = linkTable;
        netStruct.linkCapacityInErlangs = zeros(1, N-1);
        netStruct.linkLengthInKm = linkCostVector;
        netStruct.linkAttributes = cellfun(@struct, cell(1, N-1), 'UniformOutput', false);

        %% Remove routing information (now, it's obsolete)
        netStruct = removeRoutingInformation(netStruct);
        
        exitFlag = 0;
        exitMsg = 'OK!';
        
    catch e
        
        exitFlag = -1;
        exitMsg = e.message;
        netStruct = [];

    end

end

function initialNode = checkAlgorithmParameters(algorithmParameters)

    try
        assert(isfield(algorithmParameters, 'initialNode') && ~isempty(algorithmParameters.initialNode), '"initialNode" is required');
        initialNode = str2double(algorithmParameters.initialNode);
        assert(isfinite(initialNode) && initialNode > 0 && initialNode == round(initialNode), '"initialNode" must be an non-negative non-zero integer');
    catch e
        error('%s: %s', 'checkAlgorithmParameters', e.message);
    end
    
end