function [exitFlag exitMsg netStruct] = na_randomUniform(netStruct, algorithmParameters, net2planParameters)

	% na_randomUniform
	%
	% Algorithm for node assignment problems which gives a set of random nodes
	% within a given grid.
	%
	% This algorithm will remove any previous information in netStruct.
	%
    % PARAM: N | 4 | Number of nodes
    % PARAM: xmin | 0 | Minimum x-value of the grid
    % PARAM: xmax | 100 | Maximum x-value of the grid
    % PARAM: ymin | 0 | Minimum y-value of the grid
    % PARAM: ymax | 100 | Maximum y-value of the grid

    try
		[N, xmin, xmax, ymin, ymax] = checkAlgorithmParameters(algorithmParameters);
        
        nodeXYPositionTable = rand(N, 2);
        nodeXYPositionTable(:,1) = nodeXYPositionTable(:,1) * (xmax-xmin) + xmin;
        nodeXYPositionTable(:,2) = nodeXYPositionTable(:,2) * (ymax-ymin) + ymin;
        
        netStruct = struct();

        netStruct.nodeXYPositionTable = nodeXYPositionTable;
        netStruct.nodeName = cell(1,N);
        netStruct.nodeAttributes = cell(1,N);

        for nodeId = 1:N
            netStruct.nodeName{nodeId} = sprintf('Node %d', nodeId);
            netStruct.nodeAttributes{nodeId} = struct();
        end

        exitFlag = 0;
        exitMsg = 'OK!';
    
    catch e

        exitFlag = -1;
        exitMsg = e.message;
        netStruct = [];
        
    end

end

function [N, xmin, xmax, ymin, ymax] = checkAlgorithmParameters(algorithmParameters)

	try
        assert(isfield(algorithmParameters, 'N') && ~isempty(algorithmParameters.N), 'Parameter "N" is required');
        assert(isfield(algorithmParameters, 'xmin') && ~isempty(algorithmParameters.xmin), 'Parameter "xmin" is required');
        assert(isfield(algorithmParameters, 'xmax') && ~isempty(algorithmParameters.xmax), 'Parameter "xmax" is required');
        assert(isfield(algorithmParameters, 'ymin') && ~isempty(algorithmParameters.ymin), 'Parameter "ymin" is required');
        assert(isfield(algorithmParameters, 'ymax') && ~isempty(algorithmParameters.ymax), 'Parameter "ymax" is required');

        N = str2double(algorithmParameters.N);
        xmin = str2double(algorithmParameters.xmin);
        xmax = str2double(algorithmParameters.xmax);
        ymin = str2double(algorithmParameters.ymin);
        ymax = str2double(algorithmParameters.ymax);
		
		assert(isfinite(N) && N == round(N) && N > 0, '"N" must be greater than 0');
		assert(isfinite(xmin), '"xmin" must be a number');
		assert(isfinite(xmax), '"xmax" must be a number');
		assert(isfinite(ymin), '"ymin" must be a number');
		assert(isfinite(ymax), '"ymax" must be a number');
		
		assert(xmin <= xmax, '"xmax" must be greater or equal than "xmin"');
		assert(ymin <= ymax, '"ymax" must be greater or equal than "ymin"');
	catch e
		error('%s: %s', 'checkAlgorithmParameters', e.message);
	end

end