function yesNo = isBidirectionalAndSymmetric(linkTable, weightPerLink)

    % isBidirectionalAndSymmetric
    %
    % Check whether the topology is bidirectional and symmetric: same number
	% of links between each node pair in both directions (assuming
	% multi-digraphs) and same weights per direction
	%
	% A weight can be: link capacity, link length ...
    %
    % Syntax: yesNo = isBidirectionalAndSymmetric(linkTable, weightPerLink)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - weightPerLink (1xE): Row vector with a weight per link
    %
    % Output parameters:
    % - yesNo: '1' if is traffic-symmetric, '0' if not

    try
        
        %% Sanity checks
        assert(nargin == 2, 'The number of input arguments must be 2');
        assert(all(isfinite(linkTable(:))) && all(linkTable(:) == round(linkTable(:))) && ...
            ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && ...
			min(linkTable(:)) > 0, ...
            'linkTable must be a non-negative non-zero integer Ex2 matrix');
			
		E = size(linkTable, 1);
		
		assert(ndims(weightPerLink) == 2 && isequal(size(weightPerLink), [1 E]) && ...
			all(isfinite(weightPerLink)), 'weightPerLink must be a 1xE vector');
		
        %% Perform check
        yesNo = 0;

        uniqueLinkTable = unique(linkTable, 'rows');
        upwardLinks = find(diff(uniqueLinkTable, 1, 2) > 0);
        downwardLinks = find(diff(uniqueLinkTable, 1, 2) < 0);
        
        if numel(upwardLinks) ~= numel(downwardLinks), return; end
        
        for linkId = 1:numel(upwardLinks)
            
            setOfUpwardLinks = find(ismember(linkTable, uniqueLinkTable(linkId,:), 'rows'));
            setOfDownwardLinks = find(ismember(linkTable, uniqueLinkTable(linkId,[2 1]), 'rows'));
            
            if numel(setOfUpwardLinks) ~= numel(setOfDownwardLinks), return; end
            
            if ~isequal(sort(weightPerLink(setOfUpwardLinks)), ...
                    sort(weightPerLink(setOfDownwardLinks)))
               return; 
            end
            
        end
        
        yesNo = 1;
        
    catch e
        
        error('%s: %s', mfilename, e.message);
        
    end

end