function yesNo = isConnected(linkTable, sequenceOfNodes)

    % isConnected
    %
    % Check whether is possible to connect in a given node sequence using
    % a set of links
    %
    % Syntax: yesNo = isConnected(linkTable, sequenceOfNodes)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - sequenceOfNodes: Ordered sequence of nodes to find a logical path
    %
    % Output parameters:
    % - yesNo: '1' if is connected, '0' if not

    try
        %% Sanity checks
        assert(nargin == 2, 'The number of input arguments must be 2');
        assert(all(isfinite(linkTable(:))) && ...
            ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && ...
			min(linkTable(:)) > 0, ...
            'linkTable must be a non-negative non-zero integer Ex2 matrix');
        assert(all(isfinite(sequenceOfNodes(:))) && ndims(sequenceOfNodes) == 2 && size(sequenceOfNodes, 1) == 1 && ...
            all(sequenceOfNodes > 0), ['Sequence of nodes must be a row vector ' ...
            'with non-negative non-zero integer entries']);
		
        %% Perform connectivity check
		N = max(linkTable(:));
        if max(sequenceOfNodes(:)) > N
            yesNo = 0;
            return;
        end
        
        E = size(linkTable, 1);
        
        % First of all, check whether exists, at least, one incoming link
        % per the N-1 first nodes and one outgoing link per the N-1 last
        % nodes. Then check whether is possible to construct a logical ring
        % (sequenceOfNodes(1) -> sequenceOfNodes(2) -> sequenceOfNodes(3),
        % and so on)
        
        if all(ismember(sequenceOfNodes(1:end-1), linkTable(:,2)')) && ...
            all(ismember(sequenceOfNodes(2:end), linkTable(:,1)'))
            
            for originNodeIdx = 1:numel(sequenceOfNodes)-1
                
                sequenceOfLinks = shortestPath(linkTable, ones(1, E), ...
                    sequenceOfNodes(originNodeIdx), ...
                    sequenceOfNodes(originNodeIdx+1));
                
                if isempty(sequenceOfLinks)
                    yesNo = 0;
                    return;
                end
        
            end
            
        else
            
            yesNo = 0;
            return;
            
        end
        
        yesNo = 1;
		
    catch e
        
        error('%s: %s', mfilename, e.message);
        
    end

end