function [costMatrix nextHopMatrix] = floydAlgorithm(linkTable, costPerLink, N)

	% floydAlgorithm
	%
	% Perform Floyd's algorithm [1] for finding shortest path in a weighted graph
	% with positive weights.
	%
	% Input parameters:
	% - linkTable (Ex2): Table in which each row represents the origin and destination
	%   node of a link
	% - costPerLink (1xE): Positive weight of each link
	% - N: Number of nodes in the whole network
	%
	% Output parameters:
	% - costMatrix (NxN): Each entry (i,j) represents the minimum cost to go from i to j
	% - nextHopMatrix (NxN): Each entry (i,j) represents the next hop to go from i to j
	%
	% References
	%
	% [1] R.W. Floyd, "Algorithm 97: Shortest Path", Communications of the ACM
	% vol. 5, no. 6, pp. 345-345, June 1962
	

    try
		%% Sanity checks
		assert(nargin == 3, 'Requires 3 input parameters: linkTable, costPerLink, N');
        
        assert(numel(N)==1 && isfinite(N) && N==round(N) && N > 0, 'N must be greater than 0');
        
        assert(~isempty(linkTable) && all(isfinite(linkTable(:))) && ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && all(linkTable(:) > 0) && ...
            max(linkTable(:)) <= N && ...
            all(linkTable(:) == round(linkTable(:))), ...
            sprintf('linkTable must be a Ex2 matrix with each entry in range [1,%d]', N));
        
        E = size(linkTable, 1);

		assert(~isempty(costPerLink) && ndims(costPerLink) == 2 && ...
			isequal(size(costPerLink), [1 E]) && all(costPerLink >= 0) && ...
			all(isfinite(costPerLink(:))), ...
            'costPerLink must be a non-negative 1xE vector');
			
        shortestLinkMatrix = accumarray(linkTable, costPerLink, [N N], ...
            @(x) find(costPerLink == min(x),1,'first'), inf);
        
        costMatrix = accumarray(linkTable, costPerLink, [N N], ...
            @min, inf);

        costMatrix(sub2ind([N N], 1:N, 1:N)) = 0;

		if nargout > 1
            
			%% Initialize nextHopMatrix
			nextHopMatrix = kron(1:N,ones(N,1));
			nextHopMatrix = nextHopMatrix .* (1 - (nextHopMatrix == inf)) .* (1-eye(N,N));
			nextHopMatrix(nextHopMatrix == 0) = inf;
			
			for intermediateNode = 1:N
				newCostMatrix = min(costMatrix, ...
					repmat(costMatrix(:,intermediateNode), [1 N]) + ... 
					repmat(costMatrix(intermediateNode,:), [N 1]));
				
				% Extra processing to get routing matrix
				chooseAlternative = 1 - (newCostMatrix == costMatrix);
				chooseAlternative(intermediateNode, :) = 0;
				chooseAlternative(:, intermediateNode) = 0;
				
				costMatrix = newCostMatrix;
				nextHopMatrix(chooseAlternative == 1) = intermediateNode;
			end
			
			%% Change diagonal to 1:N
			nextHopMatrix(sub2ind([N N], 1:N, 1:N)) = 1:N;
			
		else
			
			for intermediateNode = 1:N
				costMatrix = min(costMatrix, ...
					repmat(costMatrix(:,intermediateNode), [1 N]) + ... 
					repmat(costMatrix(intermediateNode,:), [N 1]));
			end
			
		end
		
	catch e
	
		error('%s: %s', mfilename, e.message);
	
    end
    
end
