function [cellOfSequenceOfLinks, cellOfSequenceOfNodes, cellOfTotalCost] = kShortestPath(linkTable, costPerLink, originNode, destinationNode, K)

	% kShortestPath
	%
	% Function to obtain the K-shortest paths from originNode to
	% destinationNode. If exists parallel links between node pairs, it's
    % considered only the lower cost link among them.
    %
    % It follows the algorithm described in [1].
    %
    % Output is ordered in ascendent order, that is, the first output is
    % referred to the shortest shortest-path, and the last one is referred
    % to the largest shortest-path.
    %
    % If k < K paths are found, last (K-k) paths are returned as [].
	%
    % Syntax: [cellOfSequenceOfLinks, cellOfSequenceOfNodes, cellOfTotalCost] =
    %            kShortestPath(linkTable, costPerLink, originNode, destinationNode, K)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - costPerLink (1xE): Each entry is a non-negative weight per its
    %   corresponding link
    % - originNode: Origin node of the paths
    % - destinationNode: Destination node of the paths
    % - K: Number of shortest paths to find
    %
    % Output parameters:
    % - cellOfSequenceOfLinks {1xK}: Each entry represents a sequence of
    %   links from origin to destination
    % - cellOfSequenceOfNodes {1xK}: Each entry represents a sequence of
    %   nodes from origin to destination
    % - cellOfTotalCost {1xK}: Each entry represents the cost of that path
    %
    % References:
    % [1] J.Y. Yen, "Finding the K Shortest Loopless Paths in a Network", 
    %     Management Science vol. 17, no. 11, pp. 712-716, July 1971
    
    try
        %% Sanity checks
        assert(nargin == 5, 'The number of input parameters must be 5');
        assert(~isempty(K) && numel(K) == 1 && isfinite(K) && K > 0 && K == round(K), ...
            '"K" must be a non-negative non-zero integer');

        cellOfSequenceOfLinks = cell(1, K);
        cellOfSequenceOfNodes = cell(1, K);
        cellOfTotalCost = cell(1, K);
        
        if isempty(linkTable), return; end
        
        assert(~isempty(linkTable) && all(isfinite(linkTable(:))) && ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && all(linkTable(:) > 0) && all(linkTable(:) == round(linkTable(:))), ...
            '"linkTable" must be a Ex2 matrix with non-negative non-zero entries');
        E = size(linkTable, 1);
        assert(isequal(size(costPerLink), [1 E]) && all(isfinite(costPerLink)) && ...
            all(costPerLink >= 0), '"costPerLink" must be a non-negative 1xE vector');
        assert(~isempty(originNode) && numel(originNode) == 1 && isfinite(originNode) && originNode > 0 && originNode == round(originNode), ...
            '"originNode" must be a non-negative non-zero integer');
        assert(~isempty(destinationNode) && numel(destinationNode) == 1 && isfinite(destinationNode) && destinationNode > 0 && destinationNode == round(destinationNode), ...
            '"destinationNode" must be a non-negative non-zero integer');
        
        assert(originNode ~= destinationNode, '"originNode" and "destinationNode" must be different');
        
        %% Remove parallel links (get the lower cost link between each node pair)
        % Algorithm in [1] is devoted to simple graphs (with no parallel
        % links)
        
        [sortedLinkTable sortedLinkIndex] = sortrows([linkTable costPerLink'], 3);
        [uniqueLinkTable uniqueLinkIndex dummy] = unique(sortedLinkTable(:,1:2), 'rows', 'first');

        linkTable = uniqueLinkTable;
        costPerLink = sortedLinkTable(uniqueLinkIndex,3)';
        E = size(linkTable, 1);
        
        %% Compute k-shortest paths

        % If originNode doesn't have any outgoing link or destinationNode
        % doesn't have any incoming link obviosly a path can't be found
        if ~any(linkTable(:,1) == originNode) || ~any(linkTable(:,2) == destinationNode), return; end
        
        % K = 1, equivalent to a single shortest path
        [sequenceOfLinks sequenceOfNodes totalCost] = shortestPath(linkTable, costPerLink, originNode, destinationNode);
        
        if isempty(sequenceOfLinks), return; end
        
        cellOfSequenceOfLinks{1} = sequenceOfLinks;
        cellOfSequenceOfNodes{1} = sequenceOfNodes;
        cellOfTotalCost{1} = totalCost;
        
        pathNumber = 1;
        pathList = cell(0,3);
        
        while pathNumber < K
            
            for deviationNode = cellOfSequenceOfNodes{pathNumber}(1:end-1)
                
                positionInPath = find(cellOfSequenceOfNodes{pathNumber} == deviationNode);
                rootPath = cellOfSequenceOfNodes{pathNumber}(1:positionInPath);
                
                linksToRemove = [];
                
                for pathId = 1:pathNumber
                    if numel(cellOfSequenceOfNodes{pathId}) > positionInPath
                        if isequal(cellOfSequenceOfNodes{pathId}(1:positionInPath), rootPath)
                            linksToRemove = [linksToRemove cellOfSequenceOfLinks{pathId}(positionInPath)];
                        end
                    end
                end
                
                forbiddenNodes = cellOfSequenceOfNodes{pathNumber}(1:positionInPath-1);
                notValidLinks = ismember(linkTable(:,1), forbiddenNodes) | ismember(linkTable(:,2), forbiddenNodes);
                notValidLinks = notValidLinks | linkTable(:,2) == deviationNode;
                
                linksToRemove = [linksToRemove find(notValidLinks')];
                
                linksToTry = setdiff(1:E, linksToRemove);
                
                auxLinkTable = linkTable(linksToTry, :);
                auxCostPerLink = costPerLink(linksToTry);
                
                % Get spur path
                [sequenceOfLinks sequenceOfNodes dummy] = shortestPath(auxLinkTable, auxCostPerLink, deviationNode, destinationNode);
                if isempty(sequenceOfLinks), continue; end
                
                sequenceOfLinks = linksToTry(sequenceOfLinks);
                completeSequenceOfNodes = [cellOfSequenceOfNodes{pathNumber}(1:positionInPath-1) sequenceOfNodes];
                
                if any(cellfun(@(x) isequal(x, completeSequenceOfNodes), pathList(:,2))), continue; end
                
                pathList{end+1,1} = [cellOfSequenceOfLinks{pathNumber}(1:positionInPath-1) sequenceOfLinks];
                pathList{end,2} = completeSequenceOfNodes;
                pathList{end,3} = sum(costPerLink(pathList{end,1}));
            end
            
            if isempty(pathList), break; end

            [dummy, minTotalCost] = min([pathList{:,3}]);
            
            pathNumber = pathNumber + 1;
            cellOfSequenceOfLinks{pathNumber} = pathList{minTotalCost, 1};
            cellOfSequenceOfNodes{pathNumber} = pathList{minTotalCost, 2};
            cellOfTotalCost{pathNumber} = pathList{minTotalCost, 3};
            pathList(minTotalCost, :) = [];
            
        end
        
        %% Reconstruct paths from complete link table
        cellOfSequenceOfLinks = cellfun(@(x) sortedLinkIndex(uniqueLinkIndex(x))', cellOfSequenceOfLinks, 'UniformOutput', false);
        
    catch e
        error('%s: %s', mfilename, e.message);
    end

end