function [sequenceOfSPFLinkIds, sequenceOfSPFNodeIds, totalCost] = shortestPath(linkTable, costPerLink, ingressNode, egressNode)

	% shortestPath
	%
	% Function to obtain the shortest path from originNode to
	% destinationNode.
    %
    % If no path is found, it returns [].
    %
    % Syntax: [sequenceOfSPFLinkIds, sequenceOfSPFNodeIds, totalCost] =
    %            shortestPath(linkTable, costPerLink, ingressNode, egressNode)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - costPerLink (1xE): Each entry is a non-negative weight per its
    %   corresponding link
    % - ingressNode: Origin node of the path
    % - egressNode: Destination node of the path
    %
    % Output parameters:
    % - sequenceOfSPFLinkIds: Row vector representing the followed sequence
    %   of links
    % - sequenceOfSPFNodeIds: Row vector representing the followed sequence
    %   of nodes
    % - totalCost: Cost of the path
    
    try
        %% Sanity checks
        assert(nargin == 4, 'The number of input parameters must be 4');
        
        if isempty(linkTable)
            sequenceOfSPFLinkIds = [];
            sequenceOfSPFNodeIds = [];
            totalCost = [];
            return;
        end

        assert(~isempty(linkTable) && all(isfinite(linkTable(:))) && ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && all(linkTable(:) > 0) && all(linkTable(:) == round(linkTable(:))), ...
            '"linkTable" must be a Ex2 matrix with non-negative non-zero entries');
        E = size(linkTable, 1);
        assert(isequal(size(costPerLink), [1 E]) && all(isfinite(costPerLink)) && ...
            all(costPerLink >= 0), '"costPerLink" must be a non-negative 1xE vector');
        assert(~isempty(ingressNode) && numel(ingressNode) == 1 && isfinite(ingressNode) && ingressNode > 0 && ingressNode == round(ingressNode), ...
            '"ingressNode" must be a non-negative non-zero integer');
        assert(~isempty(egressNode) && numel(egressNode) == 1 && isfinite(egressNode) && egressNode > 0 && egressNode == round(egressNode), ...
            '"destinationNode" must be a non-negative non-zero integer');
        
        assert(ingressNode ~= egressNode, '"ingressNode" and "destinationNode" must be different');

        linkTable = [linkTable costPerLink'];

        numberNetNodes = max(max (linkTable(:,1:2)));
        numberNetNodes = max([numberNetNodes ingressNode egressNode]);

        intree = zeros (1,numberNetNodes);
        predLinkIds = -1 * ones (1,numberNetNodes);
        weightTagsPerNode = inf * ones (1,numberNetNodes);
        weightTagsPerNode (ingressNode) = 0;    

        counterOfSteps = 0;

        warning ('off' , 'MATLAB:divideByZero');
        while (intree (egressNode) ~= 1) && (counterOfSteps <= numberNetNodes)
            counterOfSteps = counterOfSteps + 1;

            % to calculate the node id with lowest tag, only among the ones in the tree
            [auxiliar,newNodeIdInTree] = min (weightTagsPerNode  ./ (1-intree));

            % If the node with lowest tag has an infinite value => nodes not in
            % the tree are not connected to the component of the ingress node, and we should stop
            if (isinf (weightTagsPerNode(newNodeIdInTree)))
                break;
            end

            intree (newNodeIdInTree) = 1;

            outgoingLinksFromNewNodeInTree = find (linkTable(:,1)==newNodeIdInTree);
            for outLinkId=outgoingLinksFromNewNodeInTree'
                neighbour = linkTable(outLinkId,2);
                if (intree (neighbour) == 0)
                    if (weightTagsPerNode (neighbour) > weightTagsPerNode (newNodeIdInTree) + linkTable (outLinkId,3))
                        weightTagsPerNode (neighbour) = weightTagsPerNode (newNodeIdInTree) + linkTable (outLinkId,3);
                        predLinkIds (neighbour) = outLinkId;
                    end
                end
            end
        end

        % convert the pred structure into a sequence of links
        sequenceOfSPFLinkIds = [];
        sequenceOfSPFNodeIds = [];

        %if the destination node is not in "intree", there was not path 
        totalCost = 0;
        if (intree (egressNode) == 1)
            node = egressNode;
            sequenceOfSPFNodeIds = [egressNode];
            while (node ~= ingressNode)
                linkToAdd = linkTable (predLinkIds(node),:);
                totalCost =  totalCost + linkToAdd(3);
                sequenceOfSPFLinkIds = [predLinkIds(node) sequenceOfSPFLinkIds];
                node = linkToAdd (1); % source of the link added
                sequenceOfSPFNodeIds = [node sequenceOfSPFNodeIds];
            end
        end

        warning ('on' , 'MATLAB:divideByZero');

    catch e
        error('%s: %s', mfilename, e.message);
    end
end