function linkTable = adjacencyMatrix2linkTable(adjacencyMatrix)

    % adjacencyMatrix2linkTable
    %
    % Function to convert a NxN adjacency matrix in a Ex2 link table where
    % each row represents the origin and destination nodes of the link.
    %
    % Syntax: linkTable = adjacencyMatrix2linkTable(adjacencyMatrix)
    %
    % Input parameters:
    % - adjacencyMatrix (NxN): Square matrix in which each entry (i,j) in
    %   shows the number of links between i and j
    %
    % Output parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link

	try
        
        %% Sanity checks
        assert(nargin == 1, 'Only 1 input parameter is required');
        assert(ndims(adjacencyMatrix) == 2 && ...
            size(adjacencyMatrix, 1) == size(adjacencyMatrix, 2), ...
            all(isfinite(adjacencyMatrix(:))) && ...
            all(adjacencyMatrix(:) >= 0) && ...
            all(adjacencyMatrix(:) == round(adjacencyMatrix(:))), ...
            'adjacencyMatrix must be a NxN non-negative integer matrix');

		%% Perform conversion
        [originNodeId destinationNodeId numberOfLinksPerNodePair] = find(adjacencyMatrix);
		
		CT = 0;
		linkTable = zeros(sum(numberOfLinksPerNodePair), 2);
		for nodePair = 1:numel(originNodeId)
			linkTable(CT + (1:numberOfLinksPerNodePair(nodePair)), 1) = originNodeId(nodePair);
			linkTable(CT + (1:numberOfLinksPerNodePair(nodePair)), 2) = destinationNodeId(nodePair);
			CT = CT + numberOfLinksPerNodePair(nodePair);
		end
		
	catch e
	
		error('%s: %s', mfilename, e.message);
		
	end
    
end