function trafficMatrix = demandTable2trafficMatrix(demandTable, offeredTrafficPerDemand, N)

    % demandTable2trafficMatrix
    %
    % Function to convert a Dx2 demand table in a NxN traffic matrix where
    % each entry (i,j) represents the units of traffic from i to j.
    %
    % If it exists more than one demand per node pair, this information is
    % lost in the traffic matrix since they are grouped in one entry.
    %
    % Syntax: trafficMatrix = demandTable2trafficMatrix(demandTable,
    % offeredTrafficPerDemand, N)
    %
    % Input parameters:
    % - demandTable (Dx2): Table in which each row represents the ingress
    %   and egress node of the demand
    % - offeredTrafficPerDemand (1xD): Row vector in which each entry
    %   represents the offered traffic for that demand
    % - N: Number of nodes in the whole network
    %
    % If demandTable = [], trafficMatrix will be a NxN zero matrix
    %
    % Output parameters:
    % - trafficMatrix (NxN): Square matrix in which each entry (i,j) in
    %   shows the units of traffic offered from i to j
    
    try
		% Sanity checks
		assert(nargin == 3, 'The number of input arguments must be 3');
		assert(numel(N) == 1 && isfinite(N) && N > 0 && N == floor(N), ...
			'N must be an integer number greater than 0')
        
        if isempty(demandTable)
            trafficMatrix = zeros(N, N);
        else
            assert(ndims(demandTable) == 2 && size(demandTable, 2) == 2 && all(isfinite(demandTable(:))) && ...
                all(demandTable(:) > 0) && all(demandTable(:) <= N) && all(demandTable(:) == round(demandTable(:))), ...
                sprintf('demandTable must be a Dx2 matrix with node identifiers between 1 and %d', N));
            assert(size(demandTable, 1) == size(offeredTrafficPerDemand, 2) && ...
                size(offeredTrafficPerDemand, 1) == 1 && ...
                all(isfinite(offeredTrafficPerDemand(:))) && ...
                all(offeredTrafficPerDemand(:) >= 0), ...
                'offeredTrafficPerDemand must be a 1xD non-negative vector');
            
            % Perform traffic matrix computation
            trafficMatrix = accumarray(demandTable, offeredTrafficPerDemand, [N N]);
        end
    catch e
		error('%s: %s', mfilename, e.message);
    end
end