function adjacencyMatrix = linkTable2adjacencyMatrix(linkTable, N)

    % linkTable2adjacencyMatrix
    %
    % Compute the adjacency matrix from a link table of a multi-digraph
    %
    % Syntax: adjacencyMatrix = linkTable2adjacencyMatrix(linkTable, N)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - N: Number of nodes in the topology
    %
    % Output parameters:
    % - adjacencyMatrix (NxN): An sparse matrix in which each position
    %   a_ij represents the number of links between nodes i and j

    try
        
        %% Sanity checks
        assert(nargin == 2, 'The number of input arguments must be 2');
        assert(numel(N) == 1 && isfinite(N) && N > 0 && N == floor(N), 'N must be greater than 0');
		
		if isempty(linkTable)
			adjacencyMatrix = zeros(N,N);
		else
			assert(ndims(linkTable) == 2 && ...
				size(linkTable, 2) == 2 && all(linkTable(:) == round(linkTable(:))) && ...
				all(isfinite(linkTable(:))) && min(linkTable(:)) > 0, ...
				'linkTable must be a non-negative non-zero integer Ex2 matrix');
			assert(max(linkTable(:)) <= N, ...
				sprintf('Node Ids in linkTable must be between 1 and %d', N));

			%% Compute adjacency matrix
			adjacencyMatrix = accumarray(linkTable, 1, [N N], [], [], true);
        end
    catch e
        error('%s: %s', mfilename, e.message);
    end

end