function incidenceMatrix = linkTable2incidenceMatrix(linkTable, N)

    % linkTable2adjacencyMatrix
    %
    % Compute the incidence matrix from a link table of a multi-digraph
    %
    % Syntax: incidenceMatrix = linkTable2incidenceMatrix(linkTable, N)
    %
    % Input parameters:
    % - linkTable (Ex2): A table in which each row represents
	%   the origin and the destination nodes of the link
    % - N: Number of nodes in the topology
    %
    % Output parameters:
    % - adjacencyMatrix (NxN): An sparse matrix in which each position
    %   a_ij represents the number of links between nodes i and j

    try
        %% Sanity checks
		assert(nargin == 2, 'A linkTable and number of nodes are required');
        assert(numel(N) == 1 && isfinite(N) && N == round(N) && N > 0, ...
            'Number of nodes must be greater than 0');
        assert(ndims(linkTable) == 2 && ...
            size(linkTable, 2) == 2 && min(linkTable(:)) > 0 && ...
            max(linkTable(:)) <= N && all(linkTable(:) == round(linkTable(:))) && ...
			all(isfinite(linkTable(:))), ...
            sprintf('linkTable must be a Ex2 matrix with node identifiers between 1 and %d', N));

        % Compute incidence matrix
		E = size(linkTable,1);
		incidenceMatrix = sparse(N, E);
		incidenceMatrix(sub2ind([N E], linkTable(:,1), (1:E)')) = 1;
		incidenceMatrix(sub2ind([N E], linkTable(:,2), (1:E)')) = -1;
	catch e
		error('%s: %s', mfilename, e.message);
    end
end