function distanceMatrix = nodeXYPositionTable2distanceMatrix(nodeXYPositionTable)

    % nodeXYPositionTable2distanceMatrix
    %
    % Compute distance matrix between each node pair from a XY-position
    % table of the nodes
    %
    % Syntax: distanceMatrix = nodeXYPositionTable2distanceMatrix(nodeXYPositionTable)
    %
    % Input parameters:
    % - nodeXYPositionTable (Nx2): A table in which each row represents the
    %   position (X,Y) in a Cartesian coordinate system
    %
    % Output parameters:
    % - distanceMatrix (NxN): A matrix in which each row i represents the
    %   distance from the node i to each other

    try
        
        %% Sanity checks
        assert(nargin == 1, 'The number of input arguments must be 1');
        assert(ndims(nodeXYPositionTable) == 2 && ...
            size(nodeXYPositionTable, 2) == 2 && all(isfinite(nodeXYPositionTable(:))), ...
            'nodeXYPositionTable must be a numeric Nx2 matrix');
        
        %% Compute distance matrix
        N = size(nodeXYPositionTable, 1);
        [i j] = find(triu(ones(N,N), 1));
        distanceMatrix = zeros(N,N);
        distanceMatrix(i + N*(j-1)) = sum((nodeXYPositionTable(i,:) - ...
            nodeXYPositionTable(j,:)).^2,2).^(1/2);
        distanceMatrix = distanceMatrix + distanceMatrix';
        
    catch e
        
        error('%s: %s', mfilename, e.message);
        
    end

end