function delta_pe = seqLinksPerPath2linkOccupancyPerPath(pathList, E)

	% seqLinksPerPath2linkOccupancyPerPath
	%
	% Function to convert a pathList into a PxE matrix in which each entry
	% (p,e) represents how many times p traverses link e
	%
	% Syntax: delta_pe = seqLinksPerPath2linkOccupancyPerPath(pathList, E)
	%
	% Input parameters:
	% - pathList {1xP}: Cell in which each entry is the sequence of links
	%   followed by path p
	% - E: Number of total links in the whole network
	%
	% Output parameters:
	% - delta_pe (PxE): Matrix in which each entry (p,e) represents 
	%   how many times p traverses link e

	try
		% Sanity checks
		assert(nargin == 2, 'The number of input arguments must be 2');
		assert(numel(E) == 1 && isfinite(E) &&  E > 0 && E == round(E), ...
			'Number of links must be greater than 0');
		assert(iscell(pathList) && ndims(pathList) == 2 && size(pathList, 1) == 1 && ...
			all(cellfun(@(x) numel(x) > 0 && all(isfinite(x(:))) && ndims(x) == 2 && ...
				size(x, 1) == 1 && all(x(:) > 0) && all(x(:) <= E), pathList)), ...
				sprintf('pathList must be a 1xP cell of arrays with linkIds in range [1, %d]', E));

        % Compute delta_pe
		P = numel(pathList);
		delta_pe = sparse(P, E);
		for pathId = 1:P
			delta_pe = delta_pe + accumarray([pathId * ones(1,numel(pathList{pathId})); pathList{pathId}]', 1, [P E], [], [], true);
		end
		
	catch e
	
		error('%s: %s', mfilename, e.message);

	end

end