function sequenceOfLinks = sequenceOfNodes2sequenceOfLinks(linkTable, sequenceOfNodes)

	% sequenceOfNodes2sequenceOfLinks
	%
	% Given a linkTable and a sequenceOfNodes, return the sequenceOfLinks
	% traversed
	%
	% In case that a node pair is connected by several links, the link considered is
	% the first one in linkTable
	%
	% Syntax: sequenceOfLinks = sequenceOfNodes2sequenceOfLinks(linkTable, sequenceOfNodes)
	%
	% Input parameters:
	% - linkTable (Ex2): Table in which each row represents the origin and destination
	%   node of the link
	% - sequenceOfNodes: Row vector with a feasible sequence of nodes
	%
	% Output parameters:
	% - sequenceOfLinks: Links traversed by sequenceOfNodes
	
    try
        %% Sanity checks
		assert(nargin == 2, 'The number of input arguments must be 2');
		assert(~isempty(linkTable) && all(isfinite(linkTable(:))) && ndims(linkTable) == 2 && ...
			size(linkTable, 2) == 2 && all(linkTable(:) > 0), 'linkTable must be a non-negative non-zero Ex2 table');
		N = max(linkTable(:));
		assert(~isempty(sequenceOfNodes) && all(isfinite(sequenceOfNodes(:))) && ndims(sequenceOfNodes) == 2 && ...
			size(sequenceOfNodes, 1) == 1 && all(sequenceOfNodes(:) > 0) && all(sequenceOfNodes(:) <= N), ...
			sprintf('sequenceOfNodes must be a valid row vector with each value in range [1, %d]', N));
		
		%% Get sequenceOfLinks
        N = numel(sequenceOfNodes);
        sequenceOfLinks = zeros(1, N-1);
        
        for sourceNodeId = 1:N-1
            linkId = find(ismember(linkTable, [sequenceOfNodes(sourceNodeId) sequenceOfNodes(sourceNodeId+1)], 'rows'), 1, 'first');
            assert(~isempty(linkId), 'Sequence of nodes not feasible');
            sequenceOfLinks(sourceNodeId) = linkId;
        end
        
    catch e
        
        error('%s: %s', mfilename, e.message);
        
    end

end