function [demandTable demandVector] = trafficMatrix2demandTable(trafficMatrix)

    % trafficMatrix2demandTable
    %
    % Function to convert a NxN traffic matrix in a Dx2 demand table and
    % 1xD demand vector where each row (a,b) in demandTable represents a traffic
    % flow from a to b, with a traffic volume according to the corresponding
    % entry in demandVector
    %
    % Syntax: [demandTable demandVector] =
    %                              trafficMatrix2demandTable(trafficMatrix)
    %
    % Input parameters:
    % - trafficMatrix (NxN): Square matrix in which each entry (i,j) in
    %   shows the units of traffic offered from i to j
    %
    % Output parameters:
    % - demandTable (Dx2): Table in which each row represents the ingress
    %   and egress node of the demand
    % - demandVector (1xD): Row vector in which each entry
    %   represents the offered traffic for that demand
    
    try
		% Sanity checks
		assert(nargin == 1, 'The number of input arguments must be 1');
		assert(ndims(trafficMatrix) == 2 && size(trafficMatrix, 1) == size(trafficMatrix, 2), ...
			'trafficMatrix must be a square matrix');
		assert(all(isfinite(trafficMatrix(:))) && all(trafficMatrix(:) >= 0), ...
			'All values must be non-negative numbers');
			
        % Transform traffic matrix into a demand table and demand vector
		[ingressNodeId egressNodeId demandVector] = find(trafficMatrix);
		
		demandTable = [ingressNodeId egressNodeId];
		demandVector = demandVector';
	catch e
		error('%s: %s', mfilename, e.message);
    end
 
end