function gradeOfService = erlangBLossProbability(numberOfServers, load)

    % erlangBLossProbability
    %
    % Return probability of call blocking in a M/M/n/n queue system
    %
    % Syntax: B = erlangBLossProbability(numberOfServers, load)
    %
    % Input parameters:
    % - numberOfServers: Number of total servers (n in a M/M/n/n system)
    % - load: Ratio of the arrival rate and service rate
    %
    % Output parameters:
    % - gradeOfService: Probability that a new call arrives, find all
    %   servers busy, and is blocked (GoS in [0,1])
    %
    % Since original Erlang's formula (1) depends on factorial of c, c is
    % constrained to a low value. This function uses an efficient
    % and exact algorithm [1] which avoids unnecessary overflows.
    %
    % B(n, load) = (load^n/n!) / sum_k=0:s (load^k/k!)                  (1)
    %
    % In addition, this non-recursive approach is faster than the iterative
    % alternative (2) and avoids reaching "RecursionLimit"
    %
    % B(n, load) = load * B(n-1, load) / (n + load * B(n-1, load))
    %                                                                   (2)
    % B(0, load) = 1
    %
    % References
    %
    % [1] S. Qiao, L. Qiao, "A Robust and Efficient Algorithm for Evaluating
    % Erlang B Formula", Technical Report CAS98-03, McMaster University
    % (Canada), October 1998
    
    try
        %% Sanity checks
        assert(nargin == 2, 'Requires two parameters, check syntax');
        assert(numel(numberOfServers) == 1 && isfinite(numberOfServers) && ...
            numberOfServers == floor(numberOfServers) && numberOfServers > 0, ...
            'Number of servers must be a non-negative non-zero integer number');
        assert(numel(load) == 1 && isfinite(load) && load >= 0, ...
			'Load must be a non-negative number');
        
        %% Perform GoS calculation
		if load < 1E-10
			gradeOfService = 0;
		else
			s = 0;
			for i=1:numberOfServers, s = (1+s) * (i/load); end
			gradeOfService = 1/(1+s);
		end
    catch e
        error('%s: %s', mfilename, e.message);
    end

end