function numberOfServers = inverseErlangB(gradeOfService, load)

    % inverseErlangB
    %
    % Return "numberOfServers" required to guarantee a certain "gradeOfService"
	% under an utilization "load" of a M/M/n/n queue system.
	%
	% Algorithm used here is presented in [1] and uses erlangBLossProbability
    %
    % Syntax: numberOfServers = inverseErlangB(gradeOfService, load)
    %
    % Input parameters:
    % - gradeOfService: Probability that a new call arrives, find all
    %   servers busy, and is blocked (GoS in [0,1])
    % - load: Ratio of the arrival rate and service rate
    %
    % Output parameters:
    % - numberOfServers: Number of total servers (n in a M/M/n/n system)
    %
    % References
    %
    % [1] S. Qiao, L. Qiao, "A Robust and Efficient Algorithm for Evaluating
    % Erlang B Formula", Technical Report CAS98-03, McMaster University
    % (Canada), October 1998
    
    try
        %% Sanity checks
        assert(nargin == 2, 'Requires two parameters, check syntax');
        assert(numel(gradeOfService) == 1 && isfinite(gradeOfService) && ...
			gradeOfService >= 0 && gradeOfService <= 1, ...
            'Grade of service must be in range [0,1]');
        assert(numel(load) == 1 && isfinite(load) && load >= 0, ...
            'Load must be a non-negative number');
        
        %% Perform computation using bisection method
		if load < 1E-10
			numberOfServers = 1;
		else
			l = 0;
			r = ceil(load);
			fR = erlangBLossProbability(r, load);
			
			while fR > gradeOfService
				l = r;
				r = r + 32;
				fR = erlangBLossProbability(r, load);
			end
			
			while r-l > 1
				mid = ceil((l+r)/2);
				fMid = erlangBLossProbability(mid, load);
				if fMid > gradeOfService
					l = mid;
				else
					r = mid;
				end
			end
			
			numberOfServers = r;
		end
    catch e
        error('%s: %s', mfilename, e.message);
    end

end