function y_e = trafficPerLink(h_d, x_dp, delta_pe)

	% trafficPerLink
	%
	% Function to compute carried traffic per link in
	% using a link-path formulation
	%
	% Syntax: y_e = trafficPerLink(h_d, x_dp, delta_pe)
	%
	% Input parameters:
	% - h_d (1xD): Vector containing offered traffic per demand
	% - x_dp (DxP): Matrix in which entry (d,p) represents the
	%   fraction of h_d carried by path p
	% - delta_pe (PxE): Matrix in which each entry (p,e) shows
	%   K if path p traverses K times link e, and otherwise, 0
	%
	% Output parameters:
	% - y_e (1xE): Carried traffic per each link

    try
		%% Sanity checks
		assert(nargin == 3, 'The number of input arguments must be 1');
		assert(ndims(h_d) == 2 && all(isfinite(h_d(:))) && size(h_d, 1) == 1 && ...
			all(h_d >= 0), 'h_d must be a non-negative 1xD vector');
		D = size(h_d, 2);
		assert(ndims(x_dp) == 2 && size(x_dp, 1) == D && ...
			all(isfinite(x_dp(:))) && all(x_dp(:) >= 0) && all(x_dp(:) <= 1), ...
			'x_dp must be a DxP matrix with each entry in range [0, 1]');
		P = size(x_dp, 2);
		assert(ndims(delta_pe) == 2 && size(delta_pe, 1) == P && ...
			all(isfinite(delta_pe(:))) && all(delta_pe(:) >= 0) && all(delta_pe(:) == round(delta_pe(:))), ...
			'delta_pe must be a non-negative integer PxE matrix');
			
		% Perform computation
        y_e = h_d * x_dp * delta_pe;
    catch e
        error('%s: %s', mfilename, e.message);
    end

end